# video_overlay_gui.py
import cv2
import os
import tkinter as tk
from tkinter import filedialog, messagebox

# === Overlay function ===
def overlay_videos(original_path, flow_path, suffix='_overlay'):
    cap1 = cv2.VideoCapture(original_path)
    cap2 = cv2.VideoCapture(flow_path)

    if not cap1.isOpened() or not cap2.isOpened():
        messagebox.showerror("Error", "Failed to open one or both videos.")
        return

    # Video properties
    fps = cap1.get(cv2.CAP_PROP_FPS)
    width = int(cap1.get(cv2.CAP_PROP_FRAME_WIDTH))
    height = int(cap1.get(cv2.CAP_PROP_FRAME_HEIGHT))

    # Prepare output path
    base, ext = os.path.splitext(original_path)
    out_path = base + suffix + ext

    # Video writer
    fourcc = cv2.VideoWriter_fourcc(*'mp4v')
    out = cv2.VideoWriter(out_path, fourcc, fps, (width, height))

    frame_count = 0

    while True:
        ret1, frame1 = cap1.read()
        ret2, frame2 = cap2.read()

        if not ret1 or not ret2:
            break

        frame2_resized = cv2.resize(frame2, (width, height))
        blended = cv2.addWeighted(frame1, 0.7, frame2_resized, 0.3, 0)
        out.write(blended)
        frame_count += 1

    cap1.release()
    cap2.release()
    out.release()
    messagebox.showinfo("Done", f"Overlay complete!\nFrames processed: {frame_count}\nSaved to:\n{out_path}")

# === GUI ===
class VideoOverlayApp:
    def __init__(self, root):
        self.root = root
        self.root.title("Overlay Flow Video on Original")
        self.root.geometry("600x300")

        self.original_path = tk.StringVar()
        self.flow_path = tk.StringVar()
        self.suffix = tk.StringVar(value="_overlay")

        # === Widgets ===
        tk.Label(root, text="1. Select Original Video:").pack(anchor='w', padx=10, pady=(10, 0))
        tk.Entry(root, textvariable=self.original_path, width=70).pack(padx=10)
        tk.Button(root, text="Browse...", command=self.select_original).pack(padx=10, pady=5)

        tk.Label(root, text="2. Select Flow Video:").pack(anchor='w', padx=10)
        tk.Entry(root, textvariable=self.flow_path, width=70).pack(padx=10)
        tk.Button(root, text="Browse...", command=self.select_flow).pack(padx=10, pady=5)

        tk.Label(root, text="3. Output Filename Suffix:").pack(anchor='w', padx=10)
        tk.Entry(root, textvariable=self.suffix, width=20).pack(padx=10, pady=5)

        tk.Button(root, text="Process Overlay", command=self.run_processing, bg="green", fg="white").pack(pady=10)

        tk.Button(root, text="Quit", command=root.quit, bg="red", fg="white").pack(pady=5)

    def select_original(self):
        path = filedialog.askopenfilename(title="Select Original Video",
                                          filetypes=[("MP4 files", "*.mp4"), ("All files", "*.*")])
        if path:
            self.original_path.set(path)

    def select_flow(self):
        path = filedialog.askopenfilename(title="Select Flow Video",
                                          filetypes=[("MP4 files", "*.mp4"), ("All files", "*.*")])
        if path:
            self.flow_path.set(path)

    def run_processing(self):
        orig = self.original_path.get()
        flow = self.flow_path.get()
        suffix = self.suffix.get()

        if not orig or not flow:
            messagebox.showerror("Missing Input", "Please select both video files.")
            return
        if not suffix.startswith("_"):
            suffix = "_" + suffix

        overlay_videos(orig, flow, suffix)

# === Run the GUI ===
if __name__ == "__main__":
    root = tk.Tk()
    app = VideoOverlayApp(root)
    root.mainloop()
