# vertical_flow_visualizer_gui.py

import cv2
import numpy as np
import os
import tkinter as tk
from tkinter import filedialog, simpledialog, messagebox

# === GUI FOR FILE SELECTION AND SUFFIX INPUT ===
root = tk.Tk()
root.withdraw()  # Hide main window

# Ask user to select a video file
input_video_path = filedialog.askopenfilename(
    title="Select Input Video",
    filetypes=[("Video Files", "*.mp4 *.avi *.mov *.mkv"), ("All Files", "*.*")]
)
if not input_video_path:
    messagebox.showerror("No file selected", "You must select a video file.")
    exit(1)

# Ask user for output suffix (default: "_yflow.mp4")
default_suffix = "_yflow.mp4"
user_suffix = simpledialog.askstring("Output Suffix", f"Enter suffix for output file (e.g., {default_suffix}):", initialvalue=default_suffix)
if not user_suffix:
    user_suffix = default_suffix

# === SETTINGS ===
colormap = cv2.COLORMAP_JET
resize_scale = 1.0

# === LOAD VIDEO ===
cap = cv2.VideoCapture(input_video_path)
if not cap.isOpened():
    raise IOError(f"Could not open video file: {input_video_path}")

# Get video properties
fps = cap.get(cv2.CAP_PROP_FPS)
frame_w = int(cap.get(cv2.CAP_PROP_FRAME_WIDTH) * resize_scale)
frame_h = int(cap.get(cv2.CAP_PROP_FRAME_HEIGHT) * resize_scale)

# Prepare output path
base, ext = os.path.splitext(input_video_path)
output_path = base + user_suffix

# Define video writer
fourcc = cv2.VideoWriter_fourcc(*'mp4v')
out = cv2.VideoWriter(output_path, fourcc, fps, (frame_w, frame_h))

# === READ FIRST FRAME ===
ret, prev = cap.read()
if not ret:
    raise RuntimeError("Failed to read first frame.")
if resize_scale != 1.0:
    prev = cv2.resize(prev, (frame_w, frame_h))
prev_gray = cv2.cvtColor(prev, cv2.COLOR_BGR2GRAY)

frame_count = 0

print("Processing...")
# === PROCESS VIDEO FRAME-BY-FRAME ===
while True:
    ret, frame = cap.read()
    if not ret:
        break
    if resize_scale != 1.0:
        frame = cv2.resize(frame, (frame_w, frame_h))
    gray = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)

    # Compute optical flow (Farneback)
    flow = cv2.calcOpticalFlowFarneback(
        prev_gray, gray, None,
        pyr_scale=0.5, levels=3, winsize=15,
        iterations=3, poly_n=5, poly_sigma=1.2, flags=0
    )

    vertical = flow[..., 1]  # extract Y motion

    # Normalize vertical motion to 0–255
    norm_v = cv2.normalize(vertical, None, 0, 255, cv2.NORM_MINMAX)
    norm_v = norm_v.astype(np.uint8)

    # Apply color map
    color_map = cv2.applyColorMap(norm_v, colormap)

    # Write frame to output
    out.write(color_map)

    prev_gray = gray
    frame_count += 1
    if frame_count % 30 == 0:
        print(f"Processed {frame_count} frames...")

# === CLEANUP ===
cap.release()
out.release()
print(f"Done. Output saved to: {output_path}")
