#vertical_flow_visualizer.py
import cv2
import numpy as np
import os

# === USER SETTINGS ===
input_video_path = 'input_video.mp4'  # ← Replace with your video file path
output_suffix = '_yflow.mp4'          # Suffix for the output video
colormap = cv2.COLORMAP_JET           # Change to other colormaps if desired
resize_scale = 1.0                    # Optional: scale down for speed

# === LOAD VIDEO ===
cap = cv2.VideoCapture(input_video_path)
if not cap.isOpened():
    raise IOError(f"Could not open video file: {input_video_path}")

# Get video properties
fps = cap.get(cv2.CAP_PROP_FPS)
frame_w = int(cap.get(cv2.CAP_PROP_FRAME_WIDTH) * resize_scale)
frame_h = int(cap.get(cv2.CAP_PROP_FRAME_HEIGHT) * resize_scale)

# Prepare output path
base, ext = os.path.splitext(input_video_path)
output_path = base + output_suffix

# Define video writer
fourcc = cv2.VideoWriter_fourcc(*'mp4v')  # Use 'XVID' or 'MJPG' if needed
out = cv2.VideoWriter(output_path, fourcc, fps, (frame_w, frame_h))

# === READ FIRST FRAME ===
ret, prev = cap.read()
if not ret:
    raise RuntimeError("Failed to read first frame.")
if resize_scale != 1.0:
    prev = cv2.resize(prev, (frame_w, frame_h))
prev_gray = cv2.cvtColor(prev, cv2.COLOR_BGR2GRAY)

frame_count = 0

print("Processing...")

# === PROCESS VIDEO FRAME-BY-FRAME ===
while True:
    ret, frame = cap.read()
    if not ret:
        break
    if resize_scale != 1.0:
        frame = cv2.resize(frame, (frame_w, frame_h))
    gray = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)

    # Compute optical flow (Farneback)
    flow = cv2.calcOpticalFlowFarneback(
        prev_gray, gray, None,
        pyr_scale=0.5, levels=3, winsize=15,
        iterations=3, poly_n=5, poly_sigma=1.2, flags=0
    )

    vertical = flow[..., 1]  # extract Y motion

    # Normalize vertical motion to 0–255
    norm_v = cv2.normalize(vertical, None, 0, 255, cv2.NORM_MINMAX)
    norm_v = norm_v.astype(np.uint8)

    # Apply color map
    color_map = cv2.applyColorMap(norm_v, colormap)

    # Write frame to output
    out.write(color_map)

    prev_gray = gray
    frame_count += 1
    if frame_count % 30 == 0:
        print(f"Processed {frame_count} frames...")

# === CLEANUP ===
cap.release()
out.release()
print(f"Done. Output saved to: {output_path}")
