# slideshow_gui.py
import os
import webbrowser
import tkinter as tk
from tkinter import filedialog, messagebox


def generate_slideshow_html(image_files):
    html_head = '''
<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Image Slideshow</title>
<style>
  body { margin: 0; background: black; overflow: hidden; }
  img { display: none; width: 100vw; height: 100vh; object-fit: contain; position: absolute; top: 0; left: 0; }
  img.active { display: block; }
</style>
</head>
<body>
'''

    html_body = ''
    for i, img in enumerate(image_files):
        cls = ' class="active"' if i == 0 else ''
        html_body += f'  <img src="{img}"{cls}>\n'

    html_script = '''
<script>
let current = 0;
const images = document.querySelectorAll("img");

function show(n) {
  images[current].classList.remove("active");
  current = (n + images.length) % images.length;
  images[current].classList.add("active");
}

document.addEventListener("keydown", (e) => {
  if (e.key === "ArrowRight") show(current + 1);
  else if (e.key === "ArrowLeft") show(current - 1);
});

document.addEventListener("click", () => {
  document.documentElement.requestFullscreen().catch(() => {});
});
</script>
</body>
</html>
'''
    return html_head + html_body + html_script


def get_images_from_directory(directory):
    supported_exts = ('.jpg', '.jpeg', '.png', '.gif', '.webp', '.bmp')
    return sorted([f for f in os.listdir(directory) if f.lower().endswith(supported_exts)])


def create_slideshow(folder, status_label):
    os.chdir(folder)
    images = get_images_from_directory(folder)
    if not images:
        messagebox.showwarning("No Images", "No supported image files found in this folder.")
        status_label.config(text="No images found.")
        return

    html = generate_slideshow_html(images)
    output_file = os.path.join(folder, "slideshow.html")
    with open(output_file, "w", encoding="utf-8") as f:
        f.write(html)

    webbrowser.open(f"file://{output_file}")
    status_label.config(text="Slideshow generated and opened.")


def select_folder_and_generate(status_label):
    folder = filedialog.askdirectory(title="Select Image Folder")
    if folder:
        status_label.config(text=f"Selected folder: {folder}")
        create_slideshow(folder, status_label)
    else:
        status_label.config(text="No folder selected.")


# GUI Setup
def main():
    root = tk.Tk()
    root.title("HTML Slideshow Generator")
    root.geometry("400x200")
    root.resizable(False, False)

    btn = tk.Button(root, text="Select Image Folder & Create Slideshow",
                    command=lambda: select_folder_and_generate(status_label),
                    font=("Arial", 12), padx=10, pady=10)
    btn.pack(pady=30)

    global status_label
    status_label = tk.Label(root, text="", font=("Arial", 10), fg="blue")
    status_label.pack()

    root.mainloop()


if __name__ == "__main__":
    main()
