#media_slideshow_gui_v2.py
import os
from pathlib import Path
import tkinter as tk
from tkinter import filedialog, messagebox

# --- Supported Extensions ---
IMAGE_EXTS = {'.jpg', '.jpeg', '.png', '.gif', '.webp'}
VIDEO_EXTS = {'.mp4', '.webm', '.ogg'}

# --- HTML Template ---
HTML_TEMPLATE = '''
<!DOCTYPE html>
<html>
img {
  max-width: 100%;
  max-height: 100%;
  object-fit: contain;
}


<head>
  <meta charset="UTF-8">
  <title>Media Slideshow</title>
  <stylehtml, body {
  margin: 0;
  height: 100%;
  background: black;
  overflow: hidden;
}
.slideshow {
  display: flex;
  justify-content: center;
  align-items: center;
  height: 100%;
}


   </style>
</head>
<body>
  <div class="container">
    <div id="mediaContainer"></div>
  </div>

  <script>
    const files = {file_list_json};
    let index = 0;

    function showMedia(idx) {{
      const container = document.getElementById('mediaContainer');
      container.innerHTML = '';

      const file = files[idx];
      const ext = file.split('.').pop().toLowerCase();

      if (['mp4', 'webm', 'ogg'].includes(ext)) {{
        const video = document.createElement('video');
        video.src = file;
        video.controls = true;
        video.autoplay = true;
        container.appendChild(video);
      }} else {{
        const img = document.createElement('img');
        img.src = file;
        container.appendChild(img);
      }}
    }}

    function next() {{
      index = (index + 1) % files.length;
      showMedia(index);
    }}

    function prev() {{
      index = (index - 1 + files.length) % files.length;
      showMedia(index);
    }}

    document.addEventListener('keydown', (event) => {{
      if (event.key === 'ArrowRight') {{
        next();
      }} else if (event.key === 'ArrowLeft') {{
        prev();
      }}
    }});

    window.onload = () => showMedia(index);
  </script>
</body>
</html>
'''

# --- Main Logic ---
def generate_slideshow_html(media_dir, html_output_path):
    media_dir = Path(media_dir)
    html_dir = Path(html_output_path).parent

    # Compute relative paths from HTML file location to media files
    media_files = [
        os.path.relpath(f.resolve(), html_dir)
        for f in sorted(media_dir.iterdir())
        if f.suffix.lower() in IMAGE_EXTS | VIDEO_EXTS
    ]

    if not media_files:
        messagebox.showwarning("No Media Found", "No images or videos found in the selected folder.")
        return

    html_content = HTML_TEMPLATE.replace('{file_list_json}', str(media_files))

    try:
        with open(html_output_path, 'w', encoding='utf-8') as f:
            f.write(html_content)
        messagebox.showinfo("Success", f"Slideshow created:\n{html_output_path}")
        os.startfile(html_dir)  # Open folder (Windows)
    except Exception as e:
        messagebox.showerror("Error", f"Failed to save HTML:\n{e}")


# --- GUI Setup ---
def select_and_generate():
    # Select media folder
    media_dir = filedialog.askdirectory(title="Select Media Folder (Images & Videos)")
    if not media_dir:
        return

    # Default HTML filename
    default_filename = "slideshow.html"
    default_dir = os.path.dirname(os.path.abspath(__file__))

    # Ask user where to save the HTML file
    html_output_path = filedialog.asksaveasfilename(
        title="Save Slideshow HTML As...",
        defaultextension=".html",
        filetypes=[("HTML files", "*.html")],
        initialdir=default_dir,
        initialfile=default_filename
    )

    if html_output_path:
        generate_slideshow_html(media_dir, html_output_path)

# --- Run GUI ---
def run_gui():
    root = tk.Tk()
    root.title("Media Slideshow Generator")

    root.geometry("400x200")
    root.resizable(False, False)

    label = tk.Label(root, text="Create a Chromecast-compatible HTML slideshow\nfrom images and videos.", wraplength=350)
    label.pack(pady=20)

    select_btn = tk.Button(root, text="Select Folder and Create Slideshow", command=select_and_generate, width=30, height=2)
    select_btn.pack(pady=10)

    quit_btn = tk.Button(root, text="Quit", command=root.quit, width=20)
    quit_btn.pack(pady=5)

    root.mainloop()

# --- Entry Point ---
if __name__ == '__main__':
    run_gui()
