# make_slideshow.py
import os
import webbrowser

def generate_slideshow_html(image_files, output_path='slideshow.html'):
    html_head = '''
<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Image Slideshow</title>
<style>
  body { margin: 0; background: black; overflow: hidden; }
  img { display: none; width: 100vw; height: 100vh; object-fit: contain; position: absolute; top: 0; left: 0; }
  img.active { display: block; }
</style>
</head>
<body>
'''

    html_body = ''
    for i, img in enumerate(image_files):
        cls = ' class="active"' if i == 0 else ''
        html_body += f'  <img src="{img}"{cls}>\n'

    html_script = '''
<script>
let current = 0;
const images = document.querySelectorAll("img");

function show(n) {
  images[current].classList.remove("active");
  current = (n + images.length) % images.length;
  images[current].classList.add("active");
}

document.addEventListener("keydown", (e) => {
  if (e.key === "ArrowRight") show(current + 1);
  else if (e.key === "ArrowLeft") show(current - 1);
});

document.addEventListener("click", () => {
  document.documentElement.requestFullscreen().catch(() => {});
});
</script>
</body>
</html>
'''

    return html_head + html_body + html_script


def get_images_from_directory(directory):
    supported_exts = ('.jpg', '.jpeg', '.png', '.gif', '.webp', '.bmp')
    return sorted([f for f in os.listdir(directory) if f.lower().endswith(supported_exts)])


def create_slideshow(directory):
    os.chdir(directory)
    images = get_images_from_directory(directory)
    if not images:
        print("No images found in directory.")
        return
    html = generate_slideshow_html(images)
    with open("slideshow.html", "w", encoding="utf-8") as f:
        f.write(html)
    webbrowser.open("slideshow.html")


# Example usage: replace path below with your image folder
if __name__ == "__main__":
    folder = input("Enter path to image folder: ").strip('"')
    if os.path.isdir(folder):
        create_slideshow(folder)
    else:
        print("Invalid folder path.")
