#make_media_slideshow.py
import os
from pathlib import Path
import tkinter as tk
from tkinter import filedialog, messagebox

# --- Constants ---
IMAGE_EXTS = {'.jpg', '.jpeg', '.png', '.gif', '.webp'}
VIDEO_EXTS = {'.mp4', '.webm', '.ogg'}

HTML_TEMPLATE = '''
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Media Slideshow</title>
  <style>
    body {{ margin: 0; background-color: black; overflow: hidden; }}
    .container {{ display: flex; align-items: center; justify-content: center; height: 100vh; }}
    img, video {{ max-width: 100%; max-height: 100%; }}
  </style>
</head>
<body>
  <div class="container">
    <div id="mediaContainer"></div>
  </div>

  <script>
    const files = {file_list_json};
    let index = 0;

    function showMedia(idx) {{
      const container = document.getElementById('mediaContainer');
      container.innerHTML = '';

      const file = files[idx];
      const ext = file.split('.').pop().toLowerCase();

      if (['mp4', 'webm', 'ogg'].includes(ext)) {{
        const video = document.createElement('video');
        video.src = file;
        video.controls = true;
        video.autoplay = true;
        container.appendChild(video);
      }} else {{
        const img = document.createElement('img');
        img.src = file;
        container.appendChild(img);
      }}
    }}

    function next() {{
      index = (index + 1) % files.length;
      showMedia(index);
    }}

    function prev() {{
      index = (index - 1 + files.length) % files.length;
      showMedia(index);
    }}

    document.addEventListener('keydown', (event) => {{
      if (event.key === 'ArrowRight') {{
        next();
      }} else if (event.key === 'ArrowLeft') {{
        prev();
      }}
    }});

    window.onload = () => showMedia(index);
  </script>
</body>
</html>
'''

# --- Main Functionality ---
def generate_slideshow_html(directory):
    p = Path(directory)
    media_files = [f.name for f in sorted(p.iterdir()) if f.suffix.lower() in IMAGE_EXTS | VIDEO_EXTS]

    if not media_files:
        messagebox.showwarning("No Media Found", "No images or videos found in the selected folder.")
        return

    html_content = HTML_TEMPLATE.replace('{file_list_json}', str(media_files))
    html_path = p / 'slideshow.html'

    with open(html_path, 'w', encoding='utf-8') as f:
        f.write(html_content)

    messagebox.showinfo("Success", f"Slideshow created:\n{html_path.name}")
    os.startfile(p)  # Open folder in Explorer (Windows only)

# --- GUI Setup ---
def select_directory():
    folder_selected = filedialog.askdirectory(title="Select a Folder Containing Images/Videos")
    if folder_selected:
        generate_slideshow_html(folder_selected)

def run_gui():
    root = tk.Tk()
    root.title("Media Slideshow Generator")

    root.geometry("400x200")
    root.resizable(False, False)

    label = tk.Label(root, text="Select a folder to create slideshow from media files", wraplength=300)
    label.pack(pady=20)

    select_btn = tk.Button(root, text="Select Folder", command=select_directory, width=20, height=2)
    select_btn.pack(pady=10)

    quit_btn = tk.Button(root, text="Quit", command=root.quit, width=20)
    quit_btn.pack(pady=5)

    root.mainloop()

# --- Run ---
if __name__ == '__main__':
    run_gui()
