# frameatatime_sidebyside_vdeo3.py
import cv2
import numpy as np

# === USER SETTINGS ===
video1_path = 'video1.mp4'
video2_path = 'video2.mp4'
max_display_width = 1600  # Resize combined image to this width if needed
font = cv2.FONT_HERSHEY_SIMPLEX

# === LOAD VIDEOS ===
cap1 = cv2.VideoCapture(video1_path)
cap2 = cv2.VideoCapture(video2_path)

frames1 = int(cap1.get(cv2.CAP_PROP_FRAME_COUNT))
frames2 = int(cap2.get(cv2.CAP_PROP_FRAME_COUNT))
frame_count = min(frames1, frames2)

frame_index = 0

# === FUNCTION TO READ A FRAME ===
def read_frame(cap, index):
    cap.set(cv2.CAP_PROP_POS_FRAMES, index)
    ret, frame = cap.read()
    return frame if ret else None

# === MAIN LOOP ===
while True:
    frame1 = read_frame(cap1, frame_index)
    frame2 = read_frame(cap2, frame_index)

    if frame1 is None or frame2 is None:
        print(f"Error reading frame {frame_index}")
        break

    # === Resize to match height ===
    h1, w1 = frame1.shape[:2]
    h2, w2 = frame2.shape[:2]
    target_height = min(h1, h2)

    def resize_keep_aspect(frame, target_height):
        h, w = frame.shape[:2]
        scale = target_height / h
        return cv2.resize(frame, (int(w * scale), target_height))

    frame1_resized = resize_keep_aspect(frame1, target_height)
    frame2_resized = resize_keep_aspect(frame2, target_height)

    # === Combine horizontally ===
    combined = np.hstack((frame1_resized, frame2_resized))

    # === Optional: resize combined to fit on screen ===
    if combined.shape[1] > max_display_width:
        scale = max_display_width / combined.shape[1]
        combined = cv2.resize(combined, (0, 0), fx=scale, fy=scale)

    # === Draw frame number ===
    label = f"Frame {frame_index + 1} / {frame_count}"
    cv2.putText(combined, label, (10, 30), font, 1, (0, 255, 255), 2)

    # === Show ===
    cv2.imshow("Frame-by-Frame Compare", combined)

    key = cv2.waitKey(0) & 0xFF
    if key == ord('q'):
        break
    elif key in [81, ord('a')]:  # Left arrow or 'a'
        if frame_index > 0:
            frame_index -= 1
    elif key in [83, ord('d')]:  # Right arrow or 'd'
        if frame_index < frame_count - 1:
            frame_index += 1

# === CLEAN UP ===
cap1.release()
cap2.release()
cv2.destroyAllWindows()
