#clock_in_frame_helper.py
import tkinter as tk
from tkinter import filedialog, messagebox
import os
import subprocess
import pysrt
import datetime

def get_fps(video_file):
    """Use ffprobe to get frames per second (fps)."""
    cmd = [
        "ffprobe", "-v", "error", "-select_streams", "v:0",
        "-show_entries", "stream=r_frame_rate",
        "-of", "default=noprint_wrappers=1:nokey=1", video_file
    ]
    result = subprocess.run(cmd, capture_output=True, text=True)
    fps_str = result.stdout.strip()
    if not fps_str or "/" not in fps_str:
        raise RuntimeError("Could not determine FPS")
    num, denom = fps_str.split("/")
    return float(num) / float(denom)

def frame_to_timecode(frame_num, fps):
    """Convert frame number to timedelta timestamp."""
    seconds = frame_num / fps
    return datetime.timedelta(seconds=seconds)

def apply_offset_to_srt(input_srt, output_srt, offset):
    """Shift all subtitle timestamps by given offset (timedelta)."""
    subs = pysrt.open(input_srt)
    for sub in subs:
        sub.start = (datetime.timedelta(
            hours=sub.start.hours,
            minutes=sub.start.minutes,
            seconds=sub.start.seconds,
            milliseconds=sub.start.milliseconds
        ) + offset)
        sub.end = (datetime.timedelta(
            hours=sub.end.hours,
            minutes=sub.end.minutes,
            seconds=sub.end.seconds,
            milliseconds=sub.end.milliseconds
        ) + offset)
    subs.save(output_srt, encoding='utf-8')

def compute_offset(wall_time_str, frame_num, fps):
    """Offset = Wall clock time - goggles subtitle frame time."""
    # parse wall clock
    parts = wall_time_str.split(":")
    if len(parts) != 3:
        raise ValueError("Wall clock must be HH:MM:SS")
    wall_td = datetime.timedelta(
        hours=int(parts[0]), minutes=int(parts[1]), seconds=int(parts[2])
    )

    # goggles time from frame number
    goggles_td = frame_to_timecode(frame_num, fps)

    return wall_td - goggles_td

def run_process():
    try:
        video_file = video_path.get()
        srt_file = srt_path.get()
        wall_clock_str = wall_time.get()
        frame_num = int(frame_number.get())

        if not os.path.exists(video_file) or not os.path.exists(srt_file):
            messagebox.showerror("Error", "Video or SRT file not found")
            return

        fps = get_fps(video_file)
        offset = compute_offset(wall_clock_str, frame_num, fps)

        # Save corrected SRT
        base, ext = os.path.splitext(srt_file)
        output_srt = f"{base}_RTC{ext}"
        apply_offset_to_srt(srt_file, output_srt, offset)

        messagebox.showinfo(
            "RTC Correction",
            f"RTC Offset: {offset}\nCorrected SRT saved as:\n{output_srt}"
        )
    except Exception as e:
        messagebox.showerror("Error", str(e))

# ---------------- GUI ----------------
root = tk.Tk()
root.title("DJI Clock-in-Frame Helper")

video_path = tk.StringVar()
srt_path = tk.StringVar()
wall_time = tk.StringVar()
frame_number = tk.StringVar()

tk.Label(root, text="Video file:").grid(row=0, column=0, sticky="e")
tk.Entry(root, textvariable=video_path, width=50).grid(row=0, column=1)
tk.Button(root, text="Browse", command=lambda: video_path.set(filedialog.askopenfilename(filetypes=[("MP4 files", "*.mp4")]))).grid(row=0, column=2)

tk.Label(root, text="Subtitle (SRT):").grid(row=1, column=0, sticky="e")
tk.Entry(root, textvariable=srt_path, width=50).grid(row=1, column=1)
tk.Button(root, text="Browse", command=lambda: srt_path.set(filedialog.askopenfilename(filetypes=[("SRT files", "*.srt")]))).grid(row=1, column=2)

tk.Label(root, text="Wall clock (HH:MM:SS):").grid(row=2, column=0, sticky="e")
tk.Entry(root, textvariable=wall_time).grid(row=2, column=1, sticky="w")

tk.Label(root, text="Frame number:").grid(row=3, column=0, sticky="e")
tk.Entry(root, textvariable=frame_number).grid(row=3, column=1, sticky="w")

tk.Button(root, text="Process", command=run_process).grid(row=4, column=1, pady=10)

root.mainloop()