# Bixler Motor Throttle Calibration Notebook
# This notebook models output power vs throttle position

import numpy as np
import matplotlib.pyplot as plt
from scipy.optimize import curve_fit
import pandas as pd

# --- Step 1: Input your measurement data here ---
# Replace with your actual data
# Throttle from -1000 to +1000
# Power_in is measured in watts (electrical input)
# RPM is optional but useful for estimating prop efficiency

# Example test data
# Throttle, Power In (W), RPM
raw_data = [
    [-1000, 0, 0],
    [-800, 3, 1500],
    [-600, 7, 2500],
    [-400, 12, 3500],
    [-200, 20, 4600],
    [0, 50, 6000],  # midpoint known point
    [200, 90, 7200],
    [400, 130, 8300],
    [600, 170, 9100],
    [800, 210, 9800],
    [1000, 250, 10500]
]

# Convert to DataFrame
cols = ["Throttle", "PowerIn_W", "RPM"]
df = pd.DataFrame(raw_data, columns=cols)

# Normalize throttle input to [-1.0, 1.0]
df["ThrottleNorm"] = df["Throttle"] / 1000.0

# --- Step 2: Define a model for fitting ---
# Quadratic model is common for ESC throttle response
def throttle_to_power(throttle, a, b):
    return a * throttle**2 + b

# Fit curve to input data
params, _ = curve_fit(throttle_to_power, df["ThrottleNorm"], df["PowerIn_W"])

# --- Step 3: Plotting the result ---
throttle_range = np.linspace(-1, 1, 200)
predicted_power = throttle_to_power(throttle_range, *params)

plt.figure(figsize=(10, 6))
plt.plot(df["ThrottleNorm"], df["PowerIn_W"], 'o', label='Measured Data')
plt.plot(throttle_range, predicted_power, '-', label=f'Fitted Model: P = {params[0]:.1f}·T² + {params[1]:.1f}')
plt.title("Throttle vs Power (Input)")
plt.xlabel("Normalized Throttle [-1 to 1]")
plt.ylabel("Power Input (Watts)")
plt.grid(True)
plt.legend()
plt.tight_layout()
plt.show()

# --- Optional: Save model parameters for flight analysis ---
print(f"Calibrated Model: Power = {params[0]:.3f} * Throttle^2 + {params[1]:.3f}")
