# Get_air_unit_and_local_time.py
import serial
import serial.tools.list_ports
import time
from datetime import datetime

BAUD_RATE = 115200

def find_air_unit_ports():
    """Scan all serial ports and return likely candidates for the Air Unit."""
    ports = serial.tools.list_ports.comports()
    candidates = []
    for port in ports:
        if "USB" in port.description or "UART" in port.description:
            candidates.append(port.device)
    return candidates

def query_air_unit_time(port):
    """Connect to a serial port and try to read the current time."""
    try:
        with serial.Serial(port, BAUD_RATE, timeout=2) as ser:
            print(f"\nConnected to Air Unit on {port}")
            time.sleep(1)
            
            # Send harmless command to read system time
            ser.write(b'date\n')
            time.sleep(0.5)
            
            response = ser.read_all().decode('utf-8', errors='ignore').strip()
            if response:
                air_unit_time = response.replace("\n", " ").strip()
                local_time = datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S UTC")
                
                print(f"Local PC time:   {local_time}")
                print(f"Air Unit time:   {air_unit_time}")
            else:
                print("No response received. Air Unit may not expose a shell over serial.")
    except serial.SerialException as e:
        print(f"Failed to connect to {port}: {e}")

def main():
    ports = find_air_unit_ports()
    if not ports:
        print("No likely Air Unit serial ports found.")
        return
    
    print("Found possible Air Unit ports:", ", ".join(ports))
    
    for port in ports:
        query_air_unit_time(port)

if __name__ == "__main__":
    main()